'use strict';

/* --------------------------------------------------------------
 create_missing_documents.js 2016-10-17
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

gx.controllers.module('overview', ['modal', 'xhr', gx.source + '/libs/info_messages'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Module Options
     *
     * @type {object}
     */
    var defaults = {};

    /**
     * Final Module Options
     *
     * @type {object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Object
     *
     * @type {object}
     */
    var module = {};

    /**
     * VPE Creation modal.
     *
     * @type {*}
     */
    var $creationModal = $('.create-modal');

    /**
     * VPE edit modal.
     *
     * @type {*}
     */
    var $editModal = $('.edit-modal');

    /**
     * VPE remove confirmation modal.
     *
     * @type {*}
     */
    var $removeConfirmationModal = $('.remove-confirmation-modal');

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Initializes the creation modal.
     *
     * @private
     */
    var _initCreationModal = function _initCreationModal() {
        var $inputs = $creationModal.find('input[type="text"]');
        var i = 0;

        for (; i < $inputs.length; i++) {
            $inputs[i].value = '';
            $($inputs[i]).parent().removeClass('has-error');
        }
        $creationModal.find('p.vpe-modal-info').first().text(jse.core.lang.translate('TEXT_INFO_INSERT_INTRO', 'products_vpe')).removeClass('text-danger');
        $creationModal.modal('show');
    };

    /**
     * Initializes the edit modal.
     *
     * @param e Event object to fetch targets id.
     * @private
     */
    var _initEditModal = function _initEditModal(e) {
        jse.libs.xhr.get({
            url: './admin.php?do=VPEAjax/getById&id=' + e.target.dataset.id
        }).done(function (r) {
            var $editModalForm = $editModal.find('form');
            var $idInput = $('<input/>').attr('type', 'hidden').attr('name', 'id');
            var $hiddenDefaultInput = void 0;
            $editModalForm.empty();

            $idInput.val(r.id.toString());
            $editModalForm.append($idInput);
            $editModal.find('p.vpe-modal-info').first().text(jse.core.lang.translate('TEXT_INFO_EDIT_INTRO', 'products_vpe')).removeClass('text-danger');

            for (var languageCode in r.names) {
                var $formGroup = $('<div/>').addClass('form-group');
                var $inputContainer = $('<div/>').addClass('col-sm-12');
                var $input = $('<input/>').attr('type', 'text').attr('data-gx-widget', 'icon_input').attr('data-lang-id', r.languageIds[languageCode]).attr('name', 'name[' + r.languageIds[languageCode] + ']').addClass('form-control').val(r.names[languageCode]);

                $inputContainer.append($input);
                $formGroup.append($inputContainer);
                $editModalForm.append($formGroup);
            }

            if (r.default) {
                $hiddenDefaultInput = $('<input/>').attr('type', 'hidden').attr('name', 'is-default').val('1');
                $editModalForm.append($hiddenDefaultInput);
            }

            $editModalForm.append(_createDefaultSwitcher(r.default));
            gx.widgets.init($editModalForm);
            $editModal.modal('show');
        });
    };

    /**
     * Creates the switcher to set the vpe default.
     *
     * @param {bool} isDefault If true, the switcher is checked.
     * @private
     */
    var _createDefaultSwitcher = function _createDefaultSwitcher(isDefault) {
        var $formGroup = $('<div/>').addClass('form-group');
        var $label = $('<label/>').addClass('col-sm-2 control-label').text(jse.core.lang.translate('TEXT_DEFAULT', 'admin_general'));
        var $checkboxContainer = $('<div/>').addClass('col-sm-10').attr('data-gx-widget', 'checkbox');
        var $checkbox = $('<input/>').attr('type', 'checkbox').attr('name', 'default').attr('title', jse.core.lang.translate('TEXT_DEFAULT', 'admin_general'));
        if (isDefault) {
            $checkbox.prop('checked', true);
        }
        $checkboxContainer.append($checkbox);

        return $formGroup.append($label).append($checkboxContainer);
    };

    /**
     * Initializes the remove confirmation modal.
     *
     * @param e Event object to fetch targets id.
     * @private
     */
    var _initRemoveConfirmationModal = function _initRemoveConfirmationModal(e) {
        $removeConfirmationModal.modal('show');
        jse.libs.xhr.get({
            url: './admin.php?do=VPEAjax/getById&id=' + e.target.dataset.id
        }).done(function (r) {
            var $info = $('.vpe-modal-remove-info');
            $info.empty();
            $('.vpe-remove-id').val(r.id);

            for (var languageCode in r.names) {
                var newContent = languageCode + ': ' + r.names[languageCode] + '<br/>';
                var oldContent = $info.html();
                $info.html(oldContent + newContent);
            }
        });
    };

    /**
     * Sends an ajax request to store a new vpe entity.
     *
     * @private
     */
    var _storeVpe = function _storeVpe() {
        if (_validateInputFields('create')) {
            jse.libs.xhr.post({
                url: './admin.php?do=VPEAjax/store',
                data: _inputData('create')
            }).done(function () {
                return _renderTable(jse.core.lang.translate('TEXT_INFO_ADD_SUCCESS', 'products_vpe'));
            });
        }
    };

    /**
     * Sends an ajax request to update a new vpe entity.
     *
     * @private
     */
    var _editVpe = function _editVpe() {
        if (_validateInputFields('edit')) {
            jse.libs.xhr.post({
                url: './admin.php?do=VPEAjax/edit',
                data: _inputData('edit')
            }).done(function () {
                return _renderTable(jse.core.lang.translate('TEXT_INFO_EDIT_SUCCESS', 'products_vpe'));
            });
        }
    };

    /**
     * Sends an ajax request to remove a new vpe entity.
     *
     * @private
     */
    var _removeVpe = function _removeVpe() {
        jse.libs.xhr.post({
            url: './admin.php?do=VPEAjax/remove',
            data: {
                id: $('.vpe-remove-id').val()
            }
        }).done(function () {
            return _renderTable(jse.core.lang.translate('TEXT_INFO_DELETE_SUCCESS', 'products_vpe'));
        });
    };

    /**
     * Renders the vpe table again.
     *
     * @private
     */
    var _renderTable = function _renderTable(msg) {
        $creationModal.modal('hide');
        $editModal.modal('hide');
        $removeConfirmationModal.modal('hide');

        if (undefined !== msg) {
            jse.libs.info_box.addSuccessMessage(msg);
        }

        jse.libs.xhr.get({
            url: './admin.php?do=VPEAjax/getData'
        }).done(function (r) {
            var i = 0;
            var $body = $('.vpe-table tbody');
            $body.empty();

            for (; i < r.data.length; i++) {
                var $row = $('<tr/>');
                var $dataCol = $('<td/>');
                var $actionsCol = $('<td/>').addClass('actions');
                var $actionsContainer = $('<div/>').addClass('pull-right action-list visible-on-hover');
                var $edit = $('<i/>').addClass('fa fa-pencil edit').attr('data-id', r.data[i].id);
                var isDefault = r.data[i].default ? ' (' + jse.core.lang.translate('TEXT_DEFAULT', 'admin_general') + ')' : '';
                var $delete = void 0;

                $actionsContainer.append($edit);

                if (!r.data[i].default) {
                    $delete = $('<i/>').addClass('fa fa-trash-o delete').attr('data-id', r.data[i].id);
                    $actionsContainer.append($delete);
                }

                $actionsCol.append($actionsContainer);
                $dataCol.text(r.data[i].names[r.languageCode] + isDefault);

                $row.append($dataCol).append($actionsCol);
                $body.append($row);
            }
            $this.find('.edit').on('click', _initEditModal);
            $this.find('.delete').on('click', _initRemoveConfirmationModal);
        });
    };

    /**
     * Validates the input fields, returns true if they are valid and false otherwise.
     *
     * @param {string} modal Modal instance, whether 'create' or 'edit'.
     * @returns {boolean}
     * @private
     */
    var _validateInputFields = function _validateInputFields(modal) {
        var $modal = modal === 'edit' ? $editModal : $creationModal;
        var $inputs = $modal.find('input[type="text"]');
        var valid = false;

        for (var i = 0; i < $inputs.length; i++) {
            if ($inputs[i].value !== '') {
                valid = true;
            }
        }

        if (!valid) {
            for (var _i = 0; _i < $inputs.length; _i++) {
                $($inputs[_i]).parent().addClass('has-error');
            }
            $modal.find('p.vpe-modal-info').first().text(jse.core.lang.translate('ERROR_INVALID_INPUT_FIELDS', 'products_vpe')).removeClass('text-info').addClass('text-danger');

            return false;
        } else {
            for (var _i2 = 0; _i2 < $inputs.length; _i2++) {
                $($inputs[_i2]).parent().removeClass('has-error');
            }
            var textConstant = modal === 'edit' ? 'TEXT_INFO_EDIT_INTRO' : 'TEXT_INFO_INSERT_INTRO';
            $modal.find('p.vpe-modal-info').first().text(jse.core.lang.translate(textConstant, 'products_vpe')).removeClass('text-danger');

            return true;
        }
    };

    /**
     * Fetches the data from the input fields.
     *
     * @param {string} modal Modal instance, whether 'create' or 'edit'.
     * @returns {{id: int, name: object}}
     * @private
     */
    var _inputData = function _inputData(modal) {
        var data = {};
        var $modal = modal === 'edit' ? $editModal : $creationModal;
        var $inputs = $modal.find('input[type="text"]');
        var $default = $modal.find('input[name="default"]');
        var $id = $modal.find('input[name="id"]').val();
        var $isDefault = $modal.find('input[name="is-default"]').val();

        for (var i = 0; i < $inputs.length; i++) {
            data[$inputs[i].getAttribute('name')] = $inputs[i].value;
        }

        if ($default.parent().hasClass('checked')) {
            data['default'] = true;
        }

        if (undefined !== $id) {
            data['id'] = $id;
        }

        if (undefined !== $isDefault) {
            data['isDefault'] = $isDefault;
        }

        return data;
    };

    // ------------------------------------------------------------------------
    // INITIALIZE
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $('#add-vpe').on('click', _initCreationModal);
        $this.find('.edit').on('click', _initEditModal);
        $this.find('.delete').on('click', _initRemoveConfirmationModal);

        // actions
        $('#create-vpe').on('click', _storeVpe);
        $('#edit-vpe').on('click', _editVpe);
        $('#remove-vpe').on('click', _removeVpe);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
